/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import com.inet.report.Area;
import com.inet.report.Element;
import com.inet.report.Engine;
import com.inet.report.Paragraph;
import com.inet.report.RDC;
import com.inet.report.Section;
import com.inet.report.Text;
import com.inet.report.TextPart;

/**
 * This class explains how to handle areas and sections with RDC.<br>
 * A report exists of at least three Area Pairs. These are the report area pair, the page area pair and the detail area.
 * Each Area Pair holds two references to an Area, except the Detail Area Pair, thats why it's called Detail Area
 * instead of Detail Area Pair. Each Area consists of one or more sections, that enables you to split the Area in single
 * parts.
 */
public class HandleAreasAndSectionsSample extends RDCSample {

    /**
     * Creates a new report and works with elements in sections and areas of this new report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //create a new Engine
            Engine eng = RDC.createEmptyEngine( exportFmt );

            //getting all Areas
            Area rhArea = eng.getArea( "RH" );//gets the report header area
            Area rfArea = eng.getArea( "RF" );//gets the report footer area
            Area phArea = eng.getArea( "PH" );//gets the page header area
            Area pfArea = eng.getArea( "PF" );//gets the page footer area
            Area dArea = eng.getArea( "D" );//gets the detail area

            //getting the sections and fill each with a static label (the name of the section)
            Section rhSec = rhArea.getSection( 0 );//gets the first report header section
            Text textRh = rhSec.addText( 2000, 100, 2000, 500 );
            Paragraph paraRh = textRh.addParagraph();
            paraRh.addTextPart( rhSec.getName() );

            Section rfSec = rfArea.getSection( 0 );//gets the first report footer section
            Text textRf = rfSec.addText( 2000, 100, 2000, 500 );
            Paragraph paraRf = textRf.addParagraph();
            paraRf.addTextPart( rfSec.getName() );

            Section phSec = phArea.getSection( 0 );//gets the first page header section
            Text textPh = phSec.addText( 2000, 100, 2000, 500 );
            Paragraph paraPh = textPh.addParagraph();
            paraPh.addTextPart( phSec.getName() );

            Section pfSec = pfArea.getSection( 0 );//gets the first page footer section
            Text textPf = pfSec.addText( 2000, 100, 2000, 500 );
            Paragraph paraPf = textPf.addParagraph();
            paraPf.addTextPart( pfSec.getName() );

            Section dSec = dArea.getSection( 0 );//gets the first detail section

            //Now we want to split the detail area in three sections (and write the name to them)
            Section d2Sec = dArea.addSection();
            Text textD2 = d2Sec.addText( 2000, 100, 2000, 500 );
            Paragraph paraD2 = textD2.addParagraph();
            paraD2.addTextPart( d2Sec.getName() );

            Section d3Sec = dArea.addSection();

            //if it wasn't a good idea with the second section
            //you can remove it
            dArea.removeSection( 1 );

            //After adding and deleting sections the names for the detail sections has changed.
            //Lets take a look:
            Text textD = dSec.addText( 2000, 100, 6000, 500 );
            Paragraph paraD = textD.addParagraph();
            TextPart tPartD = paraD.addTextPart( dSec.getName() );
            tPartD.setFontColor( RDC.COLOR_WHITE );

            Text textD3 = d3Sec.addText( 2000, 100, 2000, 500 );
            Paragraph paraD3 = textD3.addParagraph();
            paraD3.addTextPart( d3Sec.getName() );

            //set some colors, so we can see the sections
            dSec.setBackColor( 0x00FF0000 );
            d2Sec.setBackColor( 0x0000FF00 );
            d3Sec.setBackColor( 0x0000FF00 );

            //each visible element in the report is placed in a section
            //to add an new element you can use the add* methods in class Section
            dSec.addBox( 100, 100, 1000, 1000 ).setBackColor( 0x0000FF00 );

            //you can get all elements in a section with methods getElements and getElementsV in class Section
            for( Element elem: dSec.getElements()) {
                System.out.println( elem );
            }

            //if you had created two sections, which should be only one, you can merge them.
            dArea.mergeSectionBelow( 0 );

            //if you want to insert elements in according to the existing elements, you can get a snapshot
            //of all elements to avoid circles.
            Element[] elems = dSec.getElements();
            for( int i = 0; i < elems.length; i++ ) {
                Element elem = elems[i];
                Text text = dSec.addText( 3000, (i * 500) + 100, 5000, 500 );
                Paragraph para = text.addParagraph();
                para.addTextPart( elem.paramString() );
            }

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 1 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new HandleAreasAndSectionsSample().initUI();
    }
}
